<?php
/**
 * Página de administração do Berapay
 */

// Verificar se estamos no WordPress
if (!defined('ABSPATH')) {
    exit;
}

// Verificar permissões básicas
if (!function_exists('current_user_can') || !current_user_can('manage_woocommerce')) {
    wp_die('Você não tem permissão para acessar esta página.');
}

// Processar formulário
if (isset($_POST['submit']) && isset($_POST['wc_pix_gateway_nonce'])) {
    if (wp_verify_nonce($_POST['wc_pix_gateway_nonce'], 'wc_pix_gateway_settings')) {
        // Salvar configurações
        if (isset($_POST['wc_pix_gateway_api_key'])) {
            update_option('wc_pix_gateway_api_key', sanitize_text_field($_POST['wc_pix_gateway_api_key']));
        }
        if (isset($_POST['wc_pix_gateway_expiration_time'])) {
            update_option('wc_pix_gateway_expiration_time', intval($_POST['wc_pix_gateway_expiration_time']));
        }
        
        $sandbox = isset($_POST['wc_pix_gateway_sandbox']) ? 'yes' : 'no';
        update_option('wc_pix_gateway_sandbox', $sandbox);
        
        $debug = isset($_POST['wc_pix_gateway_debug']) ? 'yes' : 'no';
        update_option('wc_pix_gateway_debug', $debug);
        
        echo '<div class="notice notice-success"><p>Configurações salvas com sucesso!</p></div>';
    }
}

// Obter configurações atuais
$api_key = get_option('wc_pix_gateway_api_key', '');
$api_url = 'http://api.berapay.com.br/v1/gateway/processa_pagamento_api.php'; // URL fixa do Berapay
$sandbox = get_option('wc_pix_gateway_sandbox', 'yes');
$debug = get_option('wc_pix_gateway_debug', 'no');
$expiration_time = get_option('wc_pix_gateway_expiration_time', 30);

// Testar conexão com a API
$test_result = '';
if (isset($_POST['test_connection']) && isset($_POST['wc_pix_gateway_nonce'])) {
    if (wp_verify_nonce($_POST['wc_pix_gateway_nonce'], 'wc_pix_gateway_test')) {
        $test_result = test_api_connection($api_key, $api_url);
    }
}

// Função para testar conexão com a API
function test_api_connection($api_key, $api_url) {
    if (empty($api_key)) {
        return '<div class="test-result error">Por favor, preencha a API Key antes de testar.</div>';
    }
    
    $test_url = 'http://api.berapay.com.br/v1/gateway/processa_pagamento_api.php'; // URL fixa
           
    // Dados de teste conforme a documentação da API
    $test_data = array(
        'api-key' => $api_key,
        'name' => 'João Silva Teste',
        'cpf' => '394.480.048-69',
        'amount' => 10.00,
        'tipoPagamento' => 'PIX',
        'email' => 'teste@exemplo.com',
        'telefone' => '(11) 99999-9999'
    );        

    
    $response = wp_remote_post($test_url, array(
        'timeout' => 30,
        'headers' => array(
            'Content-Type' => 'application/json',
            'Accept' => 'application/json'
        ),
        'body' => json_encode($test_data)
    ));
    
    if (is_wp_error($response)) {
        return '<div class="test-result error">Erro na conexão: ' . $response->get_error_message() . '</div>';
    }
    
    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    $headers = wp_remote_retrieve_headers($response);
    
    // Log da resposta para debug
    if (get_option('wc_pix_gateway_debug') === 'yes') {
        error_log('PIX Gateway Test - Status: ' . $status_code);
        error_log('PIX Gateway Test - Response: ' . $body);
        error_log('PIX Gateway Test - Headers: ' . print_r($headers, true));
    }
    
    if ($status_code === 200) {
        $data = json_decode($body, true);
        if ($data && isset($data['status']) && $data['status'] === 'success') {
            return '<div class="test-result success">✅ Conexão bem-sucedida! API está funcionando corretamente.<br><br><strong>Detalhes:</strong><br>ID da Transação: ' . esc_html($data['idTransaction']) . '<br>Tipo: ' . esc_html($data['tipo']) . '</div>';
        } else {
            $error_msg = isset($data['message']) ? $data['message'] : 'Erro desconhecido';
            return '<div class="test-result error">❌ API respondeu, mas com erro: ' . esc_html($error_msg) . '</div>';
        }
    } else {
        $error_message = 'Erro na API. Status: ' . $status_code;
        
        // Tentar decodificar a resposta para obter mais detalhes
        $response_data = json_decode($body, true);
        if ($response_data && isset($response_data['message'])) {
            $error_message .= ' - ' . esc_html($response_data['message']);
        } else {
            $error_message .= ' - Resposta: ' . esc_html(substr($body, 0, 200));
        }
        
        // Adicionar informações de debug
        if (get_option('wc_pix_gateway_debug') === 'yes') {
            $error_message .= '<br><br><strong>Dados enviados:</strong><br>' . esc_html(json_encode($test_data, JSON_PRETTY_PRINT));
            $error_message .= '<br><br><strong>URL da API:</strong> ' . esc_html($test_url);
        }
        
        return '<div class="test-result error">' . $error_message . '</div>';
    }
}
?>

<div class="wrap">
    <h1>Configurações do Berapay</h1>
    
    <div class="wc-pix-gateway-admin">
        <div class="wc-pix-gateway-header">
            <div class="wc-pix-gateway-logo">
                <h2>Berapay para WooCommerce</h2>
                <p>Integração com gateway de pagamento PIX usando Asaas</p>
            </div>
            
            <div class="wc-pix-gateway-actions">
                <a href="<?php echo admin_url('admin.php?page=wc-settings&tab=checkout&section=pix_gateway'); ?>" 
                   class="button button-primary">
                    Configurações WooCommerce
                </a>
                
                <a href="https://web.berapay.com.br" target="_blank" class="button button-secondary">
                    Documentação da API
                </a>
            </div>
        </div>
        
        <div class="wc-pix-gateway-content">
            <div class="wc-pix-gateway-main">
                <form method="post" action="">
                    <?php wp_nonce_field('wc_pix_gateway_settings', 'wc_pix_gateway_nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="api_key">API Key</label>
                            </th>
                            <td>
                                <input type="text" id="api_key" name="wc_pix_gateway_api_key" 
                                       value="<?php echo esc_attr($api_key); ?>" class="regular-text" />
                                <p class="description">
                                    Sua chave de API única fornecida pelo gateway.
                                </p>
                            </td>
                        </tr>
                        
                        
                        <tr>
                            <th scope="row">
                                <label for="expiration_time">Tempo de Expiração (minutos)</label>
                            </th>
                            <td>
                                <input type="number" id="expiration_time" name="wc_pix_gateway_expiration_time" 
                                       value="<?php echo esc_attr($expiration_time); ?>" min="5" max="1440" class="small-text" />
                                <p class="description">
                                    Tempo em minutos para o PIX expirar (5-1440 minutos).
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="sandbox">Modo Sandbox</label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="sandbox" name="wc_pix_gateway_sandbox" 
                                           value="yes" <?php checked($sandbox, 'yes'); ?> />
                                    Habilitar modo de teste
                                </label>
                                <p class="description">
                                    Use para testar pagamentos sem processar valores reais.
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="debug">Modo Debug</label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="debug" name="wc_pix_gateway_debug" 
                                           value="yes" <?php checked($debug, 'yes'); ?> />
                                    Habilitar logs de debug
                                </label>
                                <p class="description">
                                    Registra informações detalhadas para debugging. Verifique os logs em WooCommerce > Status > Logs.
                                </p>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" name="submit" id="submit" 
                               class="button button-primary" 
                               value="Salvar Configurações" />
                    </p>
                </form>
                
                <!-- Teste de Conexão -->
                <div class="wc-pix-gateway-test-connection">
                    <h3>Testar Conexão com a API</h3>
                    <p>Teste se sua configuração está funcionando corretamente.</p>
                    
                    <form method="post" action="">
                        <?php wp_nonce_field('wc_pix_gateway_test', 'wc_pix_gateway_nonce'); ?>
                        <input type="hidden" name="test_connection" value="1" />
                        
                        <p class="submit">
                            <input type="submit" name="test_api" id="test_api" 
                                   class="button button-secondary" 
                                   value="Testar Conexão" />
                        </p>
                    </form>
                    
                    <?php if ($test_result): ?>
                        <div class="test-result">
                            <?php echo $test_result; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="wc-pix-gateway-sidebar">
                <div class="wc-pix-gateway-info-box">
                    <h3>Informações do Berapay</h3>
                    <ul>
                        <li><strong>Versão:</strong> 1.0.0</li>
                        <li><strong>Status:</strong> 
                            <?php if ($api_key): ?>
                                <span class="status-active">Ativo</span>
                            <?php else: ?>
                                <span class="status-inactive">Inativo</span>
                            <?php endif; ?>
                        </li>
                        <li><strong>Modo:</strong> 
                            <?php echo $sandbox === 'yes' ? 'Teste' : 'Produção'; ?>
                        </li>
                    </ul>
                </div>
                
                <div class="wc-pix-gateway-help-box">
                    <h3>Precisa de Ajuda?</h3>
                    <p>Consulte a documentação da API ou entre em contato com o suporte.</p>
                    
                    <div class="help-links">
                        <a href="https://web.berapay.com.br" target="_blank" class="button button-secondary">
                            Documentação
                        </a>
                        
                        <a href="mailto:suporte@seusite.com" class="button button-secondary">
                            Suporte
                        </a>
                    </div>
                </div>
                
                <div class="wc-pix-gateway-stats-box">
                    <h3>Estatísticas</h3>
                    <?php
                    // Contar pedidos com PIX
                    $pix_orders = array();
                    if (function_exists('wc_get_orders')) {
                        $pix_orders = wc_get_orders(array(
                            'payment_method' => 'pix_gateway',
                            'limit' => -1,
                            'return' => 'ids'
                        ));
                    }
                    
                    $total_orders = count($pix_orders);
                    $pending_orders = 0;
                    $completed_orders = 0;
                    
                    if ($total_orders > 0) {
                        foreach ($pix_orders as $order_id) {
                            $order = wc_get_order($order_id);
                            if ($order) {
                                if ($order->get_status() === 'pending') {
                                    $pending_orders++;
                                } elseif ($order->get_status() === 'completed') {
                                    $completed_orders++;
                                }
                            }
                        }
                    }
                    ?>
                    
                    <ul>
                        <li><strong>Total de Pedidos PIX:</strong> <?php echo $total_orders; ?></li>
                        <li><strong>Aguardando Pagamento:</strong> <?php echo $pending_orders; ?></li>
                        <li><strong>Pagamentos Confirmados:</strong> <?php echo $completed_orders; ?></li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.wc-pix-gateway-admin {
    margin-top: 20px;
}

.wc-pix-gateway-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #fff;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    margin-bottom: 20px;
}

.wc-pix-gateway-logo h2 {
    margin: 0 0 5px 0;
    color: #2c3e50;
}

.wc-pix-gateway-logo p {
    margin: 0;
    color: #7f8c8d;
}

.wc-pix-gateway-actions {
    display: flex;
    gap: 10px;
}

.wc-pix-gateway-content {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 20px;
}

.wc-pix-gateway-main {
    background: #fff;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.wc-pix-gateway-sidebar {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.wc-pix-gateway-info-box,
.wc-pix-gateway-help-box,
.wc-pix-gateway-stats-box {
    background: #fff;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.wc-pix-gateway-info-box h3,
.wc-pix-gateway-help-box h3,
.wc-pix-gateway-stats-box h3 {
    margin: 0 0 15px 0;
    color: #2c3e50;
    border-bottom: 2px solid #3498db;
    padding-bottom: 10px;
}

.wc-pix-gateway-info-box ul,
.wc-pix-gateway-stats-box ul {
    margin: 0;
    padding: 0;
    list-style: none;
}

.wc-pix-gateway-info-box li,
.wc-pix-gateway-stats-box li {
    margin-bottom: 10px;
    padding: 8px 0;
    border-bottom: 1px solid #f0f0f0;
}

.wc-pix-gateway-info-box li:last-child,
.wc-pix-gateway-stats-box li:last-child {
    border-bottom: none;
}

.status-active {
    color: #27ae60;
    font-weight: 600;
}

.status-inactive {
    color: #e74c3c;
    font-weight: 600;
}

.help-links {
    display: flex;
    flex-direction: column;
    gap: 10px;
    margin-top: 15px;
}

.help-links .button {
    text-align: center;
}

.wc-pix-gateway-test-connection {
    margin-top: 30px;
    padding-top: 20px;
    border-top: 2px solid #f0f0f0;
}

.test-result {
    margin-top: 15px;
    padding: 15px;
    border-radius: 6px;
    background: #f8f9fa;
    border-left: 4px solid #3498db;
}

.test-result.success {
    background: #d4edda;
    border-left-color: #28a745;
    color: #155724;
}

.test-result.error {
    background: #f8d7da;
    border-left-color: #dc3545;
    color: #721c24;
}

@media (max-width: 768px) {
    .wc-pix-gateway-header {
        flex-direction: column;
        gap: 15px;
        text-align: center;
    }
    
    .wc-pix-gateway-content {
        grid-template-columns: 1fr;
    }
    
    .wc-pix-gateway-actions {
        justify-content: center;
    }
}
</style>
